<?php
/**
 * Functions for handling how comments are displayed and used on the site. This allows more precise
 * control over their display and makes more filter and action hooks available to developers to use in their
 * customizations.
 *
 * @package WordPress
 * @subpackage Directory
 */

/* Filter the comment form defaults. */
add_filter( 'comment_form_defaults', 'supreme_comment_form_args' );
/* Add a few comment types to the allowed avatar comment types list. */
add_filter( 'get_avatar_comment_types', 'supreme_avatar_comment_types' );

/**
 * Arguments for the wp_list_comments_function() used in comments.php. Users can set up a
 * custom comments callback function by changing $callback to the custom function.  Note that
 * $style should remain 'ol' since this is hardcoded into the theme and is the semantically correct
 * element to use for listing comments.
 */
function supreme_list_comments_args() {
	/* Set the default arguments for listing comments. */
	$args = array(
		'style' => 'ol',
		'type' => 'all',
		'avatar_size' => 60,
		'callback' => 'supreme_comments_callback',
		'end-callback' => 'supreme_comments_end_callback',
	);
	/* Return the arguments and allow devs to overwrite them. */
	return apply_atomic( 'list_comments_args', $args );
}
/**
 * Uses the $comment_type to determine which comment template should be used. Once the
 * template is located, it is loaded for use. Child themes can create custom templates based off
 * the $comment_type. The comment template hierarchy is comment-$comment_type.php,
 * comment.php.
 *
 * The templates are saved in $supreme->comment_template[$comment_type], so each comment template
 * is only located once if it is needed. Following comments will use the saved template.
 *
 * @param array   $comment 				array of comment.
 * @param array   $args 				arguments of comments.
 * @param integer $depth 				number of replies.
 */
function supreme_comments_callback( $comment, $args, $depth ) {
	global $supreme;
	$GLOBALS['comment'] = $comment;
	$GLOBALS['comment_depth'] = $depth;
	/* Get the comment type of the current comment. */
	$comment_type = get_comment_type( $comment->comment_ID );
	/* Create an empty array if the comment template array is not set. */
	if ( ! isset( $supreme->comment_template ) || ! is_array( $supreme->comment_template ) ) {
		$supreme->comment_template = array();
	}
	/* Check if a template has been provided for the specific comment type.  If not, get the template. */
	if ( ! isset( $supreme->comment_template[ $comment_type ] ) ) {
		/* Create an array of template files to look for. */
		$templates = array( "comment-{$comment_type}.php" );
		/* If the comment type is a 'pingback' or 'trackback', allow the use of 'comment-ping.php'. */
		if ( 'pingback' == $comment_type || 'trackback' == $comment_type ) {
			$templates[] = 'comment-ping.php';
		}
		/* Add the fallback 'comment.php' template. */
		$templates[] = 'comment.php';
		/* Locate the comment template. */
		$template = locate_template( $templates );
		/* Set the template in the comment template array. */
		$supreme->comment_template[ $comment_type ] = $template;
	}
	/* If a template was found, load the template. */
	if ( ! empty( $supreme->comment_template[ $comment_type ] ) ) {
		require( $supreme->comment_template[ $comment_type ] );
	}
}
/**
 * Ends the display of individual comments. Uses the callback parameter for wp_list_comments().
 * Needs to be used in conjunction with supreme_comments_callback(). Not needed but used just in
 * case something is changed.
 */
function supreme_comments_end_callback() {
	echo '</li><!-- .comment -->';
}
/**
 * Displays the avatar for the comment author and wraps it in the comment author's URL if it is
 * available.  Adds a call to SUPREME_IMAGES . "/{$comment_type}.png" for the default avatars for
 * trackbacks and pingbacks.
 */
function supreme_avatar() {
	global $comment, $supreme;
	/* Make sure avatars are allowed before proceeding. */
	if ( ! get_option( 'show_avatars' ) ) {
		return false;
	}
	/* Get/set some comment variables. */
	$comment_type = get_comment_type( $comment->comment_ID );
	$author = get_comment_author( $comment->comment_ID );
	$url = get_comment_author_url( $comment->comment_ID );
	$avatar = '';
	$default_avatar = '';
	/* Get comment types that are allowed to have an avatar. */
	$avatar_comment_types = apply_filters( 'get_avatar_comment_types', array( 'comment' ) );
	/* If comment type is in the allowed list, check if it's a pingback or trackback. */
	if ( in_array( $comment_type, $avatar_comment_types ) ) {
		/* Set a default avatar for pingbacks and trackbacks. */
		$default_avatar = ( ( 'pingback' == $comment_type || 'trackback' == $comment_type ) ? trailingslashit( SUPREME_IMAGES ) . "{$comment_type}.png" : '' );
		/* Allow the default avatar to be filtered by comment type. */
		$default_avatar = apply_filters( "{$supreme->prefix}_{$comment_type}_avatar", $default_avatar );
	}
	/* Set up the avatar size. */
	$comment_list_args = supreme_list_comments_args();
	$size = ( ( $comment_list_args['avatar_size'] ) ? $comment_list_args['avatar_size'] : 60 );
	/* Get the avatar provided by the get_avatar() function. */
	$avatar = get_avatar( $comment, absint( $size ), $default_avatar, $author );
	/* If URL input, wrap avatar in hyperlink. */
	if ( ! empty( $url ) && ! empty( $avatar ) ) {
		$avatar = '<a href="' . esc_url( $url ) . '" rel="external nofollow" title="' . esc_attr( $author ) . '">' . $avatar . '</a>';
	}
	/* Display the avatar and allow it to be filtered. Note: Use the get_avatar filter hook where possible. */
	echo wp_kses_post( apply_filters( "{$supreme->prefix}_avatar", $avatar ) );
}
/**
 * Filters the WordPress comment_form() function that was added in WordPress 3.0.  This allows
 * the theme to preserve some backwards compatibility with its old comment form.  It also allows
 * users to build custom comment forms by filtering 'comment_form_defaults' in their child theme.
 *
 * @param array $args 			comments fields array.
 */
function supreme_comment_form_args( $args ) {
	global $user_identity;
	/* Get the current commenter. */
	$commenter = wp_get_current_commenter();
	/* Create the required <span> and <input> element class. */
	$req = ( ( get_option( 'require_name_email' ) ) ? ' <span class="required">' . __( '*', 'templatic' ) . '</span> ' : '' );
	$input_class = ( ( get_option( 'require_name_email' ) ) ? ' req' : '' );
	/* Sets up the default comment form fields. */
	$fields = array(
		'author' => '<p class="form-author' . esc_attr( $input_class ) . '"><label for="author">' . __( 'Name', 'templatic' ) . $req . '</label> <input type="text" class="text-input" name="author" id="author" value="' . esc_attr( $commenter['comment_author'] ) . '" size="40" /></p>',
		'email' => '<p class="form-email' . esc_attr( $input_class ) . '"><label for="email">' . __( 'Email', 'templatic' ) . $req . '</label> <input type="text" class="text-input" name="email" id="email" value="' . esc_attr( $commenter['comment_author_email'] ) . '" size="40" /></p>',
		'url' => '<p class="form-url"><label for="url">' . esc_html__( 'Website', 'templatic' ) . '</label><input type="text" class="text-input" name="url" id="url" value="' . esc_attr( $commenter['comment_author_url'] ) . '" size="40" /></p>',
	);

	/* Sets the default arguments for displaying the comment form. */
	$login_page = get_option( 'tevolution_login' );
	if ( isset( $login_page ) ) {
		$login_url = get_permalink( $login_page );
	} else {
		$login_url = wp_login_url( get_permalink() );
	}
	$args = array(
		'fields' => apply_filters( 'comment_form_default_fields', $fields ),
		'comment_field' => '<p class="form-textarea req"><label for="comment">' . __( 'Comment', 'templatic' ) . '</label><textarea name="comment" id="comment" cols="60" rows="10"></textarea></p>',
		'must_log_in' => '<p class="alert">' . esc_html__( 'You must be', 'templatic' ) . ' <a href="' . esc_url( $login_url ) . '" title="' . esc_html__( 'Log in', 'templatic' ) . '">' . esc_html__( 'logged in', 'templatic' ) . '</a>' . esc_html__( ' to post a comment.', 'templatic' ) . '</p><!-- .alert -->',
		'logged_in_as' => '<p class="log-in-out">' . esc_html__( 'Logged in as', 'templatic' ) . ' <a href="' . esc_url( get_author_posts_url( get_current_user_id() ) ) . '" title="' . esc_attr( $user_identity ) . '">' . esc_attr( $user_identity ) . '</a>. <a href="' . esc_url( wp_logout_url( get_permalink() ) ) . '" title="' . esc_html__( 'Log out of this account', 'templatic' ) . '">' . esc_html__( 'Log out &raquo;', 'templatic' ) . '</a></p><!-- .log-in-out -->',
		'comment_notes_before' => '',
		'comment_notes_after' => '',
		'id_form' => 'commentform',
		'id_submit' => 'submit',
		'title_reply' => ( 'post' == get_post_type() ) ? esc_html__( 'Leave a Reply', 'templatic' ) : esc_html__( 'Add a Review', 'templatic' ),
		'title_reply_to' => ( 'post' == get_post_type() ) ? esc_html__( 'Leave a Reply to %s', 'templatic' ) : esc_html__( 'Add a Review to %s', 'templatic' ),
		'cancel_reply_link' => esc_html__( 'Click here to cancel reply.', 'templatic' ),
		'label_submit' => esc_html__( 'Post Comment', 'templatic' ),
		'submit_button' => '<input type="submit" value="%4$s" class="%3$s" id="%2$s" name="%1$s">',
		'submit_field' => '%1$s %2$s',
	);
	/* Return the arguments for displaying the comment form. */

	return $args;
}
/**
 * Adds the 'pingback' and 'trackback' comment types to the allowed list of avatar comment types.  By
 * default, WordPress only allows the 'comment' comment type to have an avatar.
 *
 * @param array $types 		Types of comment.
 */
function supreme_avatar_comment_types( $types ) {
	/* Add the 'pingback' comment type. */
	$types[] = 'pingback';
	/* Add the 'trackback' comment type. */
	$types[] = 'trackback';
	/* Return the array of comment types. */
	return $types;
}
